
//
// This file contains code for the Shark machine learning library,
// version 2. The library is available under the GNU public license
// (GPL).
//


#ifndef _UTILS_H_
#define _UTILS_H_


#include "Array.h"

#include <algorithm>

#include <cstdio>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <fstream>
#include <cstring>
#include <sys/types.h>
#include <ctime>
using namespace std;

#define _USE_MATH_DEFINES
#include <cmath>



#ifdef __SOLARIS__
#include <values.h>
#include <float.h>
#else
#include <limits>
#endif

// largest double value
#ifndef MAXDOUBLE
#define MAXDOUBLE (std::numeric_limits< double >::max())
#endif

// smallest double value
#ifndef MINDOUBLE
#define MINDOUBLE (std::numeric_limits< double >::min())
#endif

//#ifndef round
//#define round(x) (floor((x) + 0.5))
//#endif


#ifdef _WIN32
#define finite(x) _finite(x)
#define isnan(x) _isnan(x)
#define hypot(x,y) _hypot(x,y)
typedef __int64 SharkInt64;
#else
typedef long SharkInt64;
#endif


// sqrt(pi)
#define SqrtPI     1.7724538509055160273

// sqrt(2*pi)
#define Sqrt2PI    2.50662827463100050242

// pi
#ifndef M_PI
#define M_PI       3.141592653589793238
#endif

// 2*pi
#ifndef M_2PI
#define M_2PI      6.28318530717958647692
#endif

// pi/2
#ifndef M_PI_2
#define M_PI_2     1.57079632679489661923
#endif

// e
#ifndef M_E
#define M_E        2.718281828459045235
#endif

// sqrt(e)
#define SqrtE      1.6487212707001281468


#ifdef min
#undef min
#endif
#ifdef max
#undef max
#endif
#ifdef sqr
#undef sqr
#endif

typedef void(*mylocfunc)(double *, double *);

void mexFunctionBobyqa(long int dim, long int maxfevals, double ftarget, double  *x,
	double rhoend, long int npt, double  *xl, double  *xu, double  *dx,
	double xtol_rel, double ftol_rel, double ftol_abs, mylocfunc ff);

class SharkException
{
public:
	SharkException(const char* file, int line, const char* message)
	{
		sprintf(msg, "exception in file %s in line %d error message: %s", file, line, message);
	}

	const char* what() const
	{
		return msg;
	}

protected:
	char msg[1024];
};


#define SHARKEXCEPTION(message) SharkException(__FILE__, __LINE__, message)


class RNG
{
public:
	virtual ~RNG()
	{};

	RNG(long s = 1)
	{
		seed(s);
	}

	void seed(long s = time(NULL))
	{
		//if( s == 0 ) s = time( NULL );
		initialSx = sx = (s & 0xff) + 1;
		initialSy = sy = ((s >> 8) & 0xff) + 10000;
		initialSz = sz = ((s >> 16) & 0xffff) + 3000;
	}

	void reset()
	{
		sx = initialSx;
		sy = initialSy;
		sz = initialSz;
	}

	long genLong()
	{
		const long MaxRand = 0x7fffffffL;

		return (long)(genDouble() *((double)MaxRand + 1));
	}

	double genDouble()
	{
		double rn;

		// the three congruential generators
		sx = (unsigned)(sx * 171UL % 30269UL);
		sy = (unsigned)(sy * 172UL % 30307UL);
		sz = (unsigned)(sz * 170UL % 30323UL);

		rn = sx / 30269. + sy / 30307. + sz / 30323.;

		return rn - (unsigned)rn;
	}

	void getStatus(unsigned &x, unsigned &y, unsigned &z) const
	{
		x = sx;
		y = sy;
		z = sz;
	}

	void setStatus(const unsigned x, const unsigned y, const unsigned z)
	{
		// initial state is not restored
		sx = x;
		sy = y;
		sz = z;
	}

	virtual double operator()()
	{
		return genDouble();
	}

	static RNG globalRng;

private:
	unsigned initialSx;
	unsigned initialSy;
	unsigned initialSz;
	unsigned sx, sy, sz;
};


template < class T >
class RandomVar
{
public:
	virtual ~RandomVar()
	{ }

	virtual T operator()() = 0;

	virtual double p(const T&) const = 0;

	virtual void   seed(long s)
	{
		rng.seed(s);
	}

	RandomVar& operator = (const RandomVar& r)
	{
		rng = r.rng; return *this;
	}

	virtual double entropy()
	{
		double   t;
		unsigned i;

		for (t = 0, i = monteCarloTrials; i--;) {
			t += log(p((*this)()));
		}

		return -t / monteCarloTrials;
	}

	virtual double KullbackLeibler(const RandomVar< T >& rv)
	{
		double   t;
		unsigned i;
		T        x;

		for (t = 0, i = monteCarloTrials; i--;) {
			x = (*this)();
			t += log(p(x) / rv.p(x));
		}

		return t / monteCarloTrials;
	}

	virtual double logLikelihood(const std::vector< T >& x) const
	{
		double l = 0;

		for (unsigned k = x.size(); k--;) {
			l += log(p(x[k]));
		}

		return l;
	}

	void setMonteCarlo(unsigned N = 10000)
	{
		monteCarloTrials = N;
	}

protected:
	RNG&     rng;
	unsigned monteCarloTrials;

	RandomVar(RNG& r = RNG::globalRng)
		: rng(r),
		monteCarloTrials(10000)
	{}
};

class DiscreteUniform : public RandomVar< long >
{
public:
	DiscreteUniform(long lo = 0, long hi = 1)
		: pLow(lo), pHigh(hi)
	{}

	DiscreteUniform(long lo, long hi, RNG& r)
		: RandomVar< long >(r), pLow(lo), pHigh(hi)
	{}

	long low() const
	{
		return pLow;
	}

	long high() const
	{
		return pHigh;
	}

	void low(long lo)
	{
		pLow = lo;
	}

	void high(long hi)
	{
		pHigh = hi;
	}

	long operator()(long lo, long hi)
	{
		return lo + long(rng() *(hi - lo + 1));
	}

	long operator()()
	{
		return operator()(pLow, pHigh);
	}

	double p(const long& x) const
	{
		return x >= pLow && x <= pHigh ? 1. / (pHigh - pLow + 1) : 0;
	}

protected:
	long pLow;
	long pHigh;
};


class Uniform : public RandomVar< double >
{
public:
	Uniform(double lo = 0, double hi = 1) : pLow(lo), pHigh(hi)
	{}

	Uniform(double lo, double hi, RNG& r)
		: RandomVar< double >(r), pLow(lo), pHigh(hi)
	{}

	double low() const
	{
		return pLow;
	}

	double high() const
	{
		return pHigh;
	}

	void low(double lo)
	{
		pLow = lo;
	}

	void high(double hi)
	{
		pHigh = hi;
	}

	double operator()(double lo, double hi)
	{
		return lo + rng() *(hi - lo);
	}

	double operator()()
	{
		return operator()(pLow, pHigh);
	}

	double p(const double& x) const
	{
		return x >= pLow && x < pHigh ? 1 / (pHigh - pLow) : 0;
	}

protected:
	double pLow;
	double pHigh;

};

class Normal : public RandomVar< double >
{
public:
	Normal(double mean = 0, double variance = 1)
		: pMean(mean),
		pStdDev(sqrt(variance)),
		iset(false),
		gset(0)      // to avoid uninitialized memory read warnings
	{
	}

	Normal(double mean, double variance, RNG& r)
		: RandomVar< double >(r),
		pMean(mean),
		pStdDev(sqrt(variance)),
		iset(false),
		gset(0)      // to avoid uninitialized memory read warnings
	{
	};

	void seed(long s)
	{
		RandomVar< double >::seed(s); iset = false;
	}

	double mean() const
	{
		return pMean;
	}

	double variance() const
	{
		return pStdDev * pStdDev;
	}

	void mean(double newMean)
	{
		pMean = newMean;
	}

	void variance(double newVar)
	{
		pStdDev = sqrt(newVar);
	}

	double operator()(double mean, double variance)
	{
		return operator()() *(sqrt(variance) / pStdDev)
			+ (mean - pMean);
	}

	double operator()()
	{
		double fac, rsq, v, w;

		if (!iset) {
			do {
				v = 2 * rng() - 1;
				w = 2 * rng() - 1;
				rsq = v * v + w * w;
			} while (rsq >= 1 || rsq == 0);

			fac = sqrt(-2 * log(rsq) / rsq);
			gset = v * fac;
			iset = true;

			return pMean + w * fac * pStdDev;
		}
		else {
			iset = false;
			return pMean + gset * pStdDev;
		}
	}

	double p(const double& x) const
	{
		double y = (x - pMean) / pStdDev;

		return exp(-y*y / 2) / (Sqrt2PI * pStdDev);
	}


protected:
	double pMean;
	double pStdDev;

private:
	bool   iset;
	double gset;
};

class Rng
{
public:
	static DiscreteUniform discrete;
	static Uniform         uni;
	static Normal          gauss;
	static void seed(long s);
};


class dvalue
{
public:
	double m_val;
	int	m_index;
	bool m_f;
public:
	dvalue(double val = 0, int index = 0)
	{
		put_Val(val, index);
	}
	~dvalue()
	{}

	void put_Val(double val, int index)
	{
		m_val = val;
		m_index = index;
	}
};

inline bool operator<(const dvalue & b, const dvalue &a)
{
	return b.m_val < a.m_val;
}

class dvector : public std::vector<double>
{
public:
	dvector(int Size = 0)
	{
		this->resize(Size);
	}
	~dvector(){}

	void Sort(dvector* a, dvector* i_arr, bool minimize = true)
	{
		*a = *this;
		i_arr->resize(a->size());

		vector<dvalue>	arr(this->size());
		for (int i = 0; i< arr.size(); i++)
			arr.at(i).put_Val(a->at(i), i);
		std::sort(arr.begin(), arr.end());
		if (minimize == true)
			for (int i = 0; i< arr.size(); i++)
			{
			a->at(i) = arr.at(i).m_val;
			i_arr->at(i) = arr.at(i).m_index;
			}
		else
			for (int i = 0; i< arr.size(); i++)
			{
			a->at(i) = arr.at(arr.size() - 1 - i).m_val;
			i_arr->at(i) = arr.at(arr.size() - 1 - i).m_index;
			}
	}
};

class Chromosome
{
	friend class Individual;
	friend class Population;
	friend class IndividualMOO;
	friend class PopulationMOO;

public:
	bool sameType(const Chromosome& c) const
	{
		return strcmp(typeOfAlleles(), c.typeOfAlleles()) == 0;
	}

	virtual ~Chromosome()
	{ }
	virtual const char* typeOfAlleles() const = 0;
	virtual unsigned    size() const = 0;
	virtual Chromosome& operator = (const Chromosome&) = 0;

	virtual void        resize(unsigned n) = 0;

	virtual bool operator < (const Chromosome& c) const
	{
		UNDEFINED
			return false;
	}
	virtual bool operator == (const Chromosome& c) const
	{
		UNDEFINED
			return false;
	}

	virtual void init(){}

protected:
	virtual Chromosome* clone() const = 0;
	virtual Chromosome* empty() const = 0;
};

template < class T >
class ChromosomeT_base : public Chromosome, public std::vector< T >
{
public:
	ChromosomeT_base()
	{ }
	explicit ChromosomeT_base(unsigned    l) : std::vector< T >(l)
	{ }
	ChromosomeT_base(unsigned l, const T& v) : std::vector< T >(l, v)
	{ }
	ChromosomeT_base(const std::vector< T >&   v) : std::vector< T >(v)
	{ }

	const char* typeOfAlleles() const
	{
		return typeid(T).name();
	}
	unsigned    sizeOfAlleles() const
	{
		return sizeof(T);
	}

	unsigned    size() const
	{
		return static_cast< const std::vector< T > * >(this)->size();
	}

	Chromosome& operator = (const Chromosome& c)
	{
		static_cast< std::vector< T > * >(this)->operator = (dynamic_cast< const std::vector< T >& >(c));
		return *this;
	}

	Chromosome& operator = (const std::vector< T >& c)
	{
		static_cast< std::vector< T > * >(this)->operator = (c);
		return *this;
	}

	Chromosome& operator = (const ChromosomeT_base< T >& c)
	{
		static_cast< std::vector< T > * >(this)->operator = (c);
		return *this;
	}

	Chromosome& operator = (const T& c)
	{
		for (unsigned i = this->size(); i--; (*this)[i] = c);
		return *this;
	}

	void resize(unsigned n)
	{
		if (n < size())
		{
			static_cast< std::vector< T > * >(this)->erase(this->begin() + n, this->end());
		}
		else if (n > size())
		{
			static_cast< std::vector< T > * >(this)->insert(this->end(), n - size(), T());
		}
	}

protected:
	Chromosome* clone() const
	{
		return new ChromosomeT_base< T >(*this);
	}
	Chromosome* empty() const
	{
		return new ChromosomeT_base< T >;
	}
};

template < class T >
class ChromosomeT : public ChromosomeT_base< T >
{
public:
	ChromosomeT()
	{ }
	explicit ChromosomeT(unsigned    l)
		: ChromosomeT_base< T >(l)
	{ }
	ChromosomeT(unsigned l, const T& v)
		: ChromosomeT_base< T >(l, v)
	{ }
	ChromosomeT(const std::vector< T >&   v)
		: ChromosomeT_base< T >(v)
	{ }

protected:
	Chromosome* clone() const
	{
		return new ChromosomeT< T >(*this);
	}
	Chromosome* empty() const
	{
		return new ChromosomeT< T >;
	}

};

template < class T >
class ChromosomeT_num : public ChromosomeT_base< T >
{
private:
	static void initialize(T& v, T min, T max)
	{
		if (T(0.5) == 0.5)    // continuous
			v = T(Rng::uni(double(min), double(max)));
		else // discrete
			v = T(Rng::discrete(long(min), long(max)));
	}

	static void cutOff(T& v, T min, T max)
	{
		if (v < min) v = min;
		if (v > max) v = max;
	}

public:
	ChromosomeT_num()
	{ }
	explicit ChromosomeT_num(unsigned l)
		: ChromosomeT_base< T >(l)
	{ }
	ChromosomeT_num(unsigned l, const T& v)
		: ChromosomeT_base< T >(l, v)
	{ }
	ChromosomeT_num(const std::vector< T >& v)
		: ChromosomeT_base< T >(v)
	{ }

	void initialize(T min, T max)
	{
		for (unsigned i = this->size(); i--;)
			initialize((*this)[i], min, max);
	}



	void initialize(const std::vector<T>& min,
		const std::vector<T>& max)
	{
		SIZE_CHECK(min.size() == max.size())

			((std::vector<T>*)this)->resize(min.size());

		for (unsigned i = this->size(); i--;)
			initialize((*this)[i], min[i], max[i]);
	}

	void cutOff(T min, T max)
	{
		for (unsigned i = this->size(); i--;)
			cutOff((*this)[i], min, max);
	}

	void cutOff(const Chromosome& minChrom,
		const Chromosome& maxChrom)
	{
		SIZE_CHECK(minChrom.size() == maxChrom.size())

			SIZE_CHECK((*this).size() == minChrom.size())

			if (this->size())
			{
			const std::vector< T >& min = dynamic_cast< const std::vector< T >& >(minChrom);
			const std::vector< T >& max = dynamic_cast< const std::vector< T >& >(maxChrom);

			for (unsigned i = this->size(); i--;)
				cutOff((*this)[i], min[i], max[i]);
			}
	}

	bool operator == (const Chromosome& c) const
	{
		return static_cast< const std::vector< T >& >(*this) == dynamic_cast< const std::vector< T >& >(c);
	}

	bool operator < (const Chromosome& c) const
	{
		return static_cast< const std::vector< T >& >(*this) <  dynamic_cast< const std::vector< T >& >(c);
	}
};

template < >
class ChromosomeT< double > : public ChromosomeT_num< double >
{
public:
	ChromosomeT()
	{ }
	explicit ChromosomeT(unsigned l)
		: ChromosomeT_num< double >(l)
	{ }
	ChromosomeT(unsigned l, const double& v)
		: ChromosomeT_num< double >(l, v)
	{ }
	ChromosomeT(const std::vector< double >& v)
		: ChromosomeT_num< double >(v)
	{ }


protected:
	Chromosome* clone() const
	{
		return new ChromosomeT< double >(*this);
	}
	Chromosome* empty() const
	{
		return new ChromosomeT< double >;
	}

};

class Individual : protected std::vector< Chromosome * >
{
public:
	Individual()
	{ }
	explicit Individual(unsigned n)
		: vector< Chromosome * >(n)
	{
		for (unsigned i = size(); i--;)
			*(begin() + i) = new ChromosomeT< char >;

		fitness = 0;
		scaledFitness = 0;
		evalFlg = true;
		feasible = 0;
		elitist = false;
	}
	Individual(unsigned n, const Chromosome& chrom)
		: vector< Chromosome * >(n)
	{
		for (unsigned i = size(); i--;)
			*(begin() + i) = chrom.clone();

		fitness = 0;
		scaledFitness = 0;
		evalFlg = true;
		feasible = 0;
		elitist = false;
	}
	Individual(const Chromosome& chrom0)
		: vector< Chromosome * >(1)
	{
		*(begin()) = chrom0.clone();

		fitness = 0;
		scaledFitness = 0;
		evalFlg = true;
		feasible = 0;
		elitist = false;
	}
	Individual(const vector< Chromosome * >& chrom)
		: vector< Chromosome * >(chrom)
	{
		fitness = 0;
		scaledFitness = 0;
		evalFlg = true;
		feasible = 0;
		elitist = false;
	}
	Individual(const Individual& indiv)
		: vector< Chromosome * >(indiv.size())
	{
		for (unsigned i = size(); i--;)
			*(begin() + i) = indiv[i].clone();

		fitness = indiv.fitness;
		scaledFitness = indiv.scaledFitness;
		evalFlg = indiv.evalFlg;
		feasible = indiv.feasible;
		elitist = indiv.elitist;
	}
	virtual ~Individual()
	{
		for (unsigned i = size(); i--;)
			delete *(begin() + i);
	}

	unsigned size() const
	{
		return static_cast< const std::vector< Chromosome * > * >(this)->size();
	}

	double   fitnessValue() const
	{
		return fitness;
	}
	bool     isFeasible() const
	{
		return feasible;
	}
	bool     isElitist() const
	{
		return elitist;
	}

	void     setFitness(double fit)
	{
		fitness = scaledFitness = fit;
	}
	void     setFeasible(bool f)
	{
		feasible = f;
	}

	Chromosome& operator [ ](unsigned i)
	{
		RANGE_CHECK(i < size())
			return *(*(begin() + i));
	}

	const Chromosome& operator [ ](unsigned i) const
	{
		RANGE_CHECK(i < size())
			return *(*(begin() + i));
	}

	Individual& operator = (const Individual& indiv)
	{
		unsigned i;

		for (i = size(); i--;)
			delete *(begin() + i);
		vector< Chromosome * >::operator = (indiv);
		for (i = size(); i--;)
			*(begin() + i) = indiv[i].clone();

		fitness = indiv.fitness;
		scaledFitness = indiv.scaledFitness;
		evalFlg = indiv.evalFlg;
		feasible = indiv.feasible;
		elitist = indiv.elitist;

		return *this;
	}

	void replace(unsigned i, const Chromosome& chrom);
	void insert(unsigned i, const Chromosome& chrom);
	void remove(unsigned i);
	void remove(unsigned from, unsigned to);

	template<class ChromosomeTemplate> void append(const ChromosomeTemplate& chrom)
	{
		ChromosomeTemplate* newChrom = new ChromosomeTemplate(chrom);
		std::vector< Chromosome * >::push_back(newChrom);
		newChrom->registerIndividual(*this, size() - 1);
	}

	void setEvaluationFlag()
	{
		evalFlg = true;
	}
	void clearEvaluationFlag()
	{
		evalFlg = false;
	}
	bool needEvaluation() const
	{
		return evalFlg;
	}

	bool operator == (const Individual& ind) const
	{
		if (size() == ind.size()) {
			for (unsigned i = 0; i < size(); ++i)
				if (!((*this)[i] == ind[i])) return false;

			return fitness == ind.fitness       &&
				scaledFitness == ind.scaledFitness &&
				evalFlg == ind.evalFlg       &&
				feasible == ind.feasible      &&
				elitist == ind.elitist;
		}
		return false;
	}
	bool operator < (const Individual& ind) const
	{
		if (size() == ind.size()) {
			bool less = false;

			for (unsigned i = 0; i < size(); ++i)
				if (ind[i] < (*this)[i])
					return false;
				else if (!less && (*this)[i] < ind[i])
					less = true;

			return less;
		}

		return size() < ind.size();
	}

	double   getFitness() const
	{
		return fitness;
	}

	double   getScaledFitness() const
	{
		return scaledFitness;
	}

	void     setScaledFitness(double sf)
	{
		scaledFitness = sf;
	}

	void     setEvalFlg(bool ef)
	{
		evalFlg = ef;
	}

	bool     getEvalFlg() const
	{
		return evalFlg;
	}

	bool     getFeasible() const
	{
		return feasible;
	}

	void     setElitist(bool e)
	{
		elitist = e;
	}

	bool     getElitist() const
	{
		return elitist;
	}

protected:
	double   fitness;
	double   scaledFitness;
	bool     evalFlg;
	bool     feasible;
	bool     elitist;

	friend class Population;
	friend class PopulationMOO;
	friend class IndividualMOO;
};

class IndividualMOO : public Individual
{

public:

	IndividualMOO(unsigned)
	{
		MOOFitness.resize(1);
		initializeMOOFitness(0.0);
		initializeUnpenalizedMOOFitness(0.0);
		setMOORank(0);
		setMOOShare(0.0);
		SaveMOOShareValue();
	}

	IndividualMOO()
		: Individual()
	{
		MOOFitness.resize(1);
		initializeMOOFitness(0.0);
		initializeUnpenalizedMOOFitness(0.0);
		setMOORank(0);
		setMOOShare(0.0);
		SaveMOOShareValue();
	}


	IndividualMOO(const Chromosome& chrom1)
		: Individual(chrom1)
	{
		MOOFitness.resize(1);
		initializeMOOFitness(0.0);
		initializeUnpenalizedMOOFitness(0.0);
		setMOORank(0);
		setMOOShare(0.0);
		SaveMOOShareValue();
	}

	IndividualMOO(const Individual& indiv1)
		: Individual(indiv1)
	{
		MOOFitness.resize(1);
		initializeMOOFitness(0.0);
		initializeUnpenalizedMOOFitness(0.0);
		setMOORank(0);
		setMOOShare(0.0);
		SaveMOOShareValue();
	}


	IndividualMOO(const IndividualMOO& indmoo)
		: Individual(dynamic_cast< const Individual& >(indmoo))
	{
		unsigned i;
		// copy data
		MOOFitness.resize(indmoo.MOOFitness.size());
		for (i = indmoo.MOOFitness.size(); i--;)
		{
			setMOOFitness(i, indmoo.MOOFitness[i]);
		}
		UnpenalizedMOOFitness.resize(indmoo.UnpenalizedMOOFitness.size());
		for (i = indmoo.UnpenalizedMOOFitness.size(); i--;)
		{
			setUnpenalizedMOOFitness(i, indmoo.UnpenalizedMOOFitness[i]);
		}
		setMOORank(indmoo.MOORank);
		setMOOShare(indmoo.MOOShare);
		fitness = indmoo.fitness;
		scaledFitness = indmoo.scaledFitness;
		evalFlg = indmoo.evalFlg;
		feasible = indmoo.feasible;
		elitist = indmoo.elitist;
	}

	~IndividualMOO()
	{ }

	void setNoOfObj(unsigned n)
	{
		MOOFitness.resize(n);
		UnpenalizedMOOFitness.resize(n);
		initializeMOOFitness(0.0);
		initializeUnpenalizedMOOFitness(0.0);
	}

	unsigned getNoOfObj()
	{
		return MOOFitness.size();
	}

	void setMOORank(unsigned n)
	{
		MOORank = n;
	}

	unsigned getMOORank()
	{
		return MOORank;
	}

	void setMOOShare(double n)
	{
		MOOShare = n;
	}
	void SaveMOOShareValue()
	{
		MOOShareValue = MOOShare;
	}

	double getMOOShare()
	{
		return MOOShare;
	}


	void setID(int val)	{ ID = val; }
	int getID()			{ return ID; }

	void setMOOFitness(unsigned nof, double fit)
	{
		RANGE_CHECK(nof < MOOFitness.size())
			MOOFitness[nof] = fit;
	}
	double getMOOFitness(unsigned nof)
	{
		RANGE_CHECK(nof < MOOFitness.size())
			return MOOFitness[nof];
	}

	void setMOOFitnessValues(double f0)
	{
		RANGE_CHECK(0 < MOOFitness.size())
			MOOFitness[0] = f0;
	}

	void setMOOFitnessValues(double f0, double f1)
	{
		RANGE_CHECK(1 < MOOFitness.size())
			MOOFitness[0] = f0;
		MOOFitness[1] = f1;
	}

	void setMOOFitnessValues(vector< double > fit)
	{
		RANGE_CHECK(fit.size() == MOOFitness.size())
			for (unsigned i = fit.size(); i--;)
			{
			MOOFitness[i] = fit[i];
			}
	}

	std::vector< double > getMOOFitnessValues()
	{
		return MOOFitness;
	}

	void initializeMOOFitness(double x)
	{
		for (unsigned i = MOOFitness.size(); i--;)
		{
			MOOFitness[i] = x;
		}
	}

	void setUnpenalizedMOOFitness(unsigned nof, double fit)
	{
		RANGE_CHECK(nof < UnpenalizedMOOFitness.size())
			UnpenalizedMOOFitness[nof] = fit;
	}
	double getUnpenalizedMOOFitness(unsigned nof)
	{
		RANGE_CHECK(nof < UnpenalizedMOOFitness.size())
			return UnpenalizedMOOFitness[nof];
	}
	void setUnpenalizedMOOFitnessValues(double f0)
	{
		RANGE_CHECK(0 < UnpenalizedMOOFitness.size())
			UnpenalizedMOOFitness[0] = f0;
	}
	void setUnpenalizedMOOFitnessValues(double f0, double f1)
	{
		RANGE_CHECK(1 < UnpenalizedMOOFitness.size())
			UnpenalizedMOOFitness[0] = f0;
		UnpenalizedMOOFitness[1] = f1;
	}

	void setUnpenalizedMOOFitnessValues(vector< double >& fit)
	{
		RANGE_CHECK(fit.size() < UnpenalizedMOOFitness.size())
			for (unsigned i = fit.size(); i--;)
			{
			UnpenalizedMOOFitness[i] = fit[i];
			}
	}
	std::vector< double >& getUnpenalizedMOOFitnessValues()
	{
		return UnpenalizedMOOFitness;
	}
	void initializeUnpenalizedMOOFitness(double x)
	{
		for (unsigned i = UnpenalizedMOOFitness.size(); i--;)
		{
			UnpenalizedMOOFitness[i] = x;
		}
	}

	Chromosome& operator [ ](unsigned i)
	{
		return Individual::operator [ ](i);
	}
	const Chromosome& operator [ ](unsigned i) const
	{
		return Individual::operator [ ](i);
	}
	IndividualMOO& operator = (const IndividualMOO& indmoo)
	{
		unsigned i;

		for (i = size(); i--;)
			delete *(begin() + i);

		vector< Chromosome * >::operator = (indmoo);

		for (i = size(); i--;)
			*(begin() + i) = indmoo[i].clone();

		if (MOOFitness.size() != indmoo.MOOFitness.size())
			setNoOfObj(indmoo.MOOFitness.size());

		fitness = indmoo.fitness;
		scaledFitness = indmoo.scaledFitness;
		evalFlg = indmoo.evalFlg;
		feasible = indmoo.feasible;
		elitist = indmoo.elitist;
		MOORank = indmoo.MOORank;
		MOOShare = indmoo.MOOShare;

		//for (i = getNoOfObj(); i--;)
		//{
		//	MOOFitness[i] = indmoo.MOOFitness[i];
		//}
		MOOFitness = indmoo.MOOFitness;
		ID = indmoo.ID;

		for (i = getNoOfObj(); i--;)
		{
			UnpenalizedMOOFitness[i] = indmoo.UnpenalizedMOOFitness[i];
		}


		return *this;
	}

	bool operator == (const IndividualMOO& indmoo) const
	{
		unsigned i;
		if (size() == indmoo.size())
		{
			for (i = size(); i--;)
			{
				if (!((*this)[i] == indmoo[i]))
				{
					return false;
				}
			}
			if (MOOFitness.size() != indmoo.MOOFitness.size() || UnpenalizedMOOFitness.size() != indmoo.UnpenalizedMOOFitness.size())
			{
				return false;
			}

			for (i = MOOFitness.size(); i--;)
			{
				if (!(MOOFitness[i] == indmoo.MOOFitness[i]))
				{
					return false;
				}
			}

			for (i = UnpenalizedMOOFitness.size(); i--;)
			{
				if (!(UnpenalizedMOOFitness[i] == indmoo.UnpenalizedMOOFitness[i]))
				{
					return false;
				}
			}

			if (fitness == indmoo.fitness       &&
				scaledFitness == indmoo.scaledFitness &&
				evalFlg == indmoo.evalFlg       &&
				feasible == indmoo.feasible      &&
				elitist == indmoo.elitist       &&
				MOORank == indmoo.MOORank       &&
				MOOShare == indmoo.MOOShare)
			{
				return true;
			}
		}
		return false;
	}
	// TO-IM-104
	IndividualMOO& operator = (const Individual& ind)
	{
		unsigned i;

		for (i = size(); i--;)
			delete *(begin() + i);

		vector< Chromosome * >::operator = (ind);

		for (i = size(); i--;)
			*(begin() + i) = ind[i].clone();

		setNoOfObj(1);
		initializeMOOFitness(0);
		initializeUnpenalizedMOOFitness(0);

		MOORank = 0;
		MOOShare = 0;

		return *this;
	}

	void replace(unsigned, const Chromosome&);
	void insert(unsigned, const Chromosome&);
	void append(const Chromosome& chrom)
	{
		vector< Chromosome * >::push_back(chrom.clone());
	}

	void remove(unsigned);
	void remove(unsigned, unsigned);

	template<class ChromosomeTemplate> void append(const ChromosomeTemplate& chrom);

	double aggregation(const std::vector< double >&);
	double simplesum();

protected:
	std::vector< double > MOOFitness;
	std::vector< double > UnpenalizedMOOFitness;
	unsigned         MOORank;
	double           MOOShare;
	double           MOOShareValue;
	int				ID;
};

class Population : protected std::vector< Individual * >
{
public:
	Population()
	{
		ascending = false;
	}
	explicit Population(unsigned n)
		: vector< Individual * >(n)
	{
		for (unsigned i = size(); i--;)
			*(begin() + i) = new Individual;
		ascending = false;
	}

	Population(const vector< Individual * >& indvec)
		: vector< Individual * >(indvec)
	{
		ascending = false;
	}
	virtual ~Population()
	{
		for (unsigned i = size(); i--;) delete *(begin() + i);
	}

	unsigned          size() const
	{
		return static_cast< const std::vector< Individual * > * >(this)->size();
	}

	void              resize(unsigned n)
	{
		unsigned i, s = size();
		if (n < s) {
			for (i = n; i < s; ++i)
				delete *(begin() + i);
			vector< Individual * >::erase(begin() + n, end());
		}
		else if (n > size()) {
			vector< Individual * >::insert(end(), n - s, (Individual *)NULL);
			if (s > 0)
				for (i = s; i < n; ++i)
					*(begin() + i) = new Individual(*(*(begin())));
			else
				for (i = s; i < n; ++i)
					*(begin() + i) = new Individual;
		}
	}

	void              setMaximize()
	{
		ascending = false;
	}
	void              setMinimize()
	{
		ascending = true;
	}

	bool              ascendingFitness() const
	{
		return ascending;
	}

	Individual&       operator [ ](unsigned i)
	{
		RANGE_CHECK(i < size())
			return *(*(begin() + i));
	}

	const Individual& operator [ ](unsigned i) const
	{
		RANGE_CHECK(i < size())
			return *(*(begin() + i));
	}

	Population        operator()(unsigned from, unsigned to) const
	{
		RANGE_CHECK(from <= to && to < size())
			return Population(std::vector< Individual * >(
			begin() + from, begin() + to + 1));
	}

	Population&       operator = (const Individual& ind)
	{
		for (unsigned i = size(); i--;)
			(*this)[i] = ind;

		return *this;
	}

	Population&       operator = (const Population& pop)
	{
		if (this == &pop) return *this;

		unsigned i;

		if (size() != pop.size()) {
			for (i = size(); i--;)
				delete *(begin() + i);
			vector< Individual * >::operator = (pop);
			for (i = size(); i--;)
				*(begin() + i) = new Individual(pop[i]);
		}
		else
			for (i = size(); i--;)
				(*this)[i] = pop[i];

		return *this;
	}

	void              swap(unsigned i, unsigned j)
	{
		RANGE_CHECK(i < size() && j < size())
			std::swap(*(begin() + i), *(begin() + j));
	}

protected:
	unsigned index;
	bool     ascending;

	Individual& worst(Individual&, Individual&) const;

	void        selectInit();
};

class myeig
{
public:
	void eigensymm
		(
		const Array2D< double >& A,
		Array2D< double >& G,
		Array< double >& l
		)
	{
		SIZE_CHECK(A.ndim() == 2);
		unsigned int n = A.dim(0);
		SIZE_CHECK(A.dim(1) == n);

		G.resize(A, false);
		l.resize(n, false);

		Array<double> od(n);

		eigensymm(A, G, l, od);
	}

	void eigensymm
		(
		const Array2D< double >& amatA,
		Array2D< double >& vmatA,
		Array  < double >& dvecA,
		Array  < double >& odvecA
		)
	{
		SIZE_CHECK(amatA.dim(0) == amatA.dim(1))

			vmatA.resize(amatA, false);
		dvecA.resize(amatA.dim(0), false);
		odvecA.resize(amatA.dim(0), false);

		const unsigned maxIterC = 50;

		unsigned n = dvecA.nelem();

		unsigned i, j, k, l, m;
		double   b, c, f, g, h, hh, p, r, s, scale;

		//
		// special case n = 1
		//
		if (n == 1) {
			vmatA(0, 0) = 1;
			dvecA(0) = amatA(0, 0);
			return;
		}

		for (i = 0; i < n; i++) {
			for (j = 0; j <= i; j++) {
				vmatA(i, j) = amatA(i, j);
			}
		}

		for (i = n; i-- > 1;) {
			h = scale = 0.0;

			if (i > 1) {
				for (k = 0; k < i; k++) {
					scale += fabs(vmatA(i, k));
				}
			}

			if (scale == 0.0) {
				odvecA(i) = vmatA(i, i - 1);
			}
			else {
				for (k = 0; k < i; k++) {
					vmatA(i, k) /= scale;
					h += vmatA(i, k) * vmatA(i, k);
				}

				f = vmatA(i, i - 1);
				g = f > 0 ? -sqrt(h) : sqrt(h);
				odvecA(i) = scale * g;
				h -= f * g;
				vmatA(i, i - 1) = f - g;
				f = 0.0;

				for (j = 0; j < i; j++) {
					vmatA(j, i) = vmatA(i, j) / (scale * h);
					g = 0.0;

					// form element of a*u
					for (k = 0; k <= j; k++) {
						g += vmatA(j, k) * vmatA(i, k);
					}

					for (k = j + 1; k < i; k++) {
						g += vmatA(k, j) * vmatA(i, k);
					}

					f += (odvecA(j) = g / h) * vmatA(i, j);
				}

				hh = f / (h + h);

				for (j = 0; j < i; j++) {
					f = vmatA(i, j);
					g = odvecA(j) - hh * f;
					odvecA(j) = g;

					for (k = 0; k <= j; k++) {
						vmatA(j, k) -= f * odvecA(k) + g * vmatA(i, k);
					}
				}

				for (k = i; k--;) {
					vmatA(i, k) *= scale;
				}
			}

			dvecA(i) = h;
		}

		dvecA(0) = odvecA(0) = 0.0;

		for (i = 0; i < n; i++) {
			if (dvecA(i)) {
				for (j = 0; j < i; j++) {
					g = 0.0;

					for (k = 0; k < i; k++) {
						g += vmatA(i, k) * vmatA(k, j);
					}

					for (k = 0; k < i; k++) {
						vmatA(k, j) -= g * vmatA(k, i);
					}
				}
			}

			dvecA(i) = vmatA(i, i);
			vmatA(i, i) = 1.0;

			for (j = 0; j < i; j++) {
				vmatA(i, j) = vmatA(j, i) = 0.0;
			}
		}

		if (n <= 1) {
			return;
		}

		for (i = 1; i < n; i++) {
			odvecA(i - 1) = odvecA(i);
		}

		odvecA(n - 1) = 0.0;

		for (l = 0; l < n; l++) {
			j = 0;

			do {
				// look for small sub-diagonal element
				for (m = l; m < n - 1; m++) {
					s = fabs(dvecA(m)) + fabs(dvecA(m + 1));
					if (fabs(odvecA(m)) + s == s) {
						break;
					}
				}

				p = dvecA(l);

				if (m != l) {
					if (j++ == maxIterC)
						throw SHARKEXCEPTION("too many iterations in eigendecomposition");

					// form shift
					g = (dvecA(l + 1) - p) / (2.0 * odvecA(l));
					r = sqrt(g * g + 1.0);
					g = dvecA(m) - p + odvecA(l) / (g + ((g) > 0 ? fabs(r) : -fabs(r)));
					s = c = 1.0;
					p = 0.0;

					for (i = m; i-- > l;) {
						f = s * odvecA(i);
						b = c * odvecA(i);

						if (fabs(f) >= fabs(g)) {
							c = g / f;
							r = sqrt(c * c + 1.0);
							odvecA(i + 1) = f * r;
							s = 1.0 / r;
							c *= s;
						}
						else {
							s = f / g;
							r = sqrt(s * s + 1.0);
							odvecA(i + 1) = g * r;
							c = 1.0 / r;
							s *= c;
						}

						g = dvecA(i + 1) - p;
						r = (dvecA(i) - g) * s + 2.0 * c * b;
						p = s * r;
						dvecA(i + 1) = g + p;
						g = c * r - b;

						// form vector
						for (k = 0; k < n; k++) {
							f = vmatA(k, i + 1);
							vmatA(k, i + 1) = s * vmatA(k, i) + c * f;
							vmatA(k, i) = c * vmatA(k, i) - s * f;
						}
					}

					dvecA(l) -= p;
					odvecA(l) = g;
					odvecA(m) = 0.0;
				}
			} while (m != l);
		}

		eigensort(vmatA, dvecA);

		for (j = n; j--;) {
			s = 0.0;
			for (i = n; i--;) {
				s += vmatA(i, j) * vmatA(i, j);
			}
			s = sqrt(s);

			for (i = n; i--;) {
				vmatA(i, j) /= s;
			}
		}
	}

	void eigensort
		(
		Array2D< double >& vmatA,
		Array  < double >& dvecA
		)
	{
		SIZE_CHECK
			(
			dvecA.ndim() == 1 &&
			dvecA.dim(0) == vmatA.dim(0) &&
			dvecA.dim(0) == vmatA.dim(1)
			)

			unsigned n = dvecA.nelem();
		unsigned i, j, l;
		double t;

		for (i = 0; i < n - 1; i++) {
			t = dvecA(l = i);
			for (j = i + 1; j < n; j++) {
				if (dvecA(j) >= t) {
					t = dvecA(l = j);
				}
			}

			if (l != i) {
				dvecA(l) = dvecA(i);
				dvecA(i) = t;
				for (j = 0; j < n; j++) {
					t = vmatA(j, i);
					vmatA(j, i) = vmatA(j, l);
					vmatA(j, l) = t;
				}
			}
		}
	}
};

struct MyRew
{
	int iter;
	double rew;
	int isParentRew;
};

class ChromosomeCMA : public ChromosomeT< double >
{
public:
	ChromosomeCMA()
	{};
	explicit ChromosomeCMA(unsigned l) : ChromosomeT< double >(l)
	{};
	ChromosomeCMA(unsigned l, const double& v) : ChromosomeT< double >(l, v)
	{};
	ChromosomeCMA(const std::vector< double >& v) : ChromosomeT< double >(v)
	{};
	~ChromosomeCMA()
	{};

	void init(unsigned             dimension,
		std::vector<double > stdv,
		double               initialSigma,
		ChromosomeT<double> &MinInit,
		ChromosomeT<double> &MaxInit,
		int                  noOffspring = 1,
		double               lower = 0.)
	{
		unsigned i;
		myeig eig;
		n = dimension;
		sigma = initialSigma;

		(*this).resize(n);

		pc.resize(n);
		C.resize(n, n, false);
		B.resize(n, n, false);
		eigenvalues.resize(n);
		z.resize(n);
		lastStep.resize(n);

		initialize(MinInit, MaxInit);

		lambda = noOffspring;
		cc = 2. / (2. + n);
		ccov = 2. / (n * n + 6.);

		lambdaSucc = 0.;
		noOffspringSinceUpdate = 0;

		lowerBound = lower;

		psPrime = 1. / (5. + sqrt((double)lambda) / 2.);
		ps = psPrime;

		cp = (psPrime * lambda) / (2 + psPrime * lambda);

		d = 1 + n / (lambda * 2);

		for (i = 0; i < n; i++) {
			pc(i) = 0.;
			for (j = 0; j < n; j++) {
				if (i != j) C(i, j) = 0;
				else C(i, j) = stdv[i] * stdv[i];
			}
		}

		eig.eigensymm(C, B, eigenvalues);
		needCovarianceUpdate = false;
	}

	void init(unsigned dimension,
		double   initialSigma,
		double   MinInit,
		double   MaxInit,
		int      lambda = 1)
	{
		std::vector<double> stdv(dimension);
		ChromosomeT<double> chromeMinInit(dimension), chromeMaxInit(dimension);

		for (i = 0; i < dimension; i++) {
			stdv[i] = 1;
			chromeMinInit[i] = MinInit;
			chromeMaxInit[i] = MaxInit;
		}
		init(dimension, stdv, initialSigma, chromeMinInit, chromeMaxInit, lambda);
	}

	Array<double> calcMean()
	{
		return (*this);
	}

	void mutate3()
	{
		Array<double> x(n);
		for (int i = 0; i<n; i++)
			x(i) = (*this)[i];
		Array<double> x2;
		x2 = x;
		bool correct = false;
		int ntry = 0;
		while (correct == false)
		{
			ntry++;
			correct = true;
			x2 = x;
			for (i = 0; i < n; i++) z(i) = Rng::gauss(0, 1);
			lastStep = 0;

			for (i = 0; i < n; i++)
				for (j = 0; j < n; j++)
					lastStep(i) += B(i, j) * sqrt(fabs(eigenvalues(j))) * z(j);

			for (i = 0; i < n; i++)
			{
				x2(i) += sigma * lastStep(i);
				if ((x(i) < 0) || (x(i) > 1))
					correct = false;
			}
			if (ntry > 1000)
				break;
		}
		for (int i = 0; i<n; i++)
			(*this)[i] = x2(i);
		needCovarianceUpdate = true;
	};

	void mutate()
	{
		for (i = 0; i < n; i++) z(i) = Rng::gauss(0, 1);
		lastStep = 0;

		for (i = 0; i < n; i++)
			for (j = 0; j < n; j++)
				lastStep(i) += B(i, j) * sqrt(fabs(eigenvalues(j))) * z(j);

		for (i = 0; i < n; i++)
			(*this)[i] += sigma * lastStep(i);

		needCovarianceUpdate = true;
	};

	void mutateClose(vector< vector<double> >* pops)
	{
		int ntry = 100;
		int itry = 0;
		bool correct = false;
		while (correct == false)
		{
			itry++;
			for (i = 0; i < n; i++) z(i) = Rng::gauss(0, 1);
			lastStep = 0;

			for (i = 0; i < n; i++)
				for (j = 0; j < n; j++)
					lastStep(i) += B(i, j) * sqrt(fabs(eigenvalues(j))) * z(j);

			vector<double>	oldx, newx;
			double distToNew = 0;
			for (int i = 0; i<n; i++)
			{
				oldx.push_back((*this)[i]);
				newx.push_back((*this)[i] + sigma * lastStep(i));
				distToNew += pow(sigma * lastStep(i), 2.0);
			}
			distToNew = sqrt(distToNew);

			double distMin = 1e+10;
			for (int i = 0; i<pops->size(); i++)
			{
				vector<double>* pIndI = &pops->at(i);
				double dist = 0;
				for (int j = 0; j<pIndI->size(); j++)
					dist += pow(pIndI->at(j) - newx.at(j), 2.0);
				dist = sqrt(dist);
				if (dist < distMin)	distMin = dist;
			}

			if (distToNew < distMin)	correct = true;
			else						correct = false;
			if (itry > ntry)			correct = true;

			if (correct)
				for (i = 0; i < n; i++)
					(*this)[i] += sigma * lastStep(i);
		}
		needCovarianceUpdate = true;
	};

	void putSigma(double newVal)
	{
		sigma = newVal;
	}

	void updateGlobalStepsize()
	{
		updateGlobalStepsize(lambdaSucc / noOffspringSinceUpdate);
		lambdaSucc = 0.;
		noOffspringSinceUpdate = 0;
	}

	void updateLambdaSucc(bool better)
	{
		noOffspringSinceUpdate++;
		if (better) lambdaSucc++;
	}

	void updateGlobalStepsize(double nsucc)
	{
		ps = (1 - cp) * ps + cp * (nsucc / lambda);
		sigma *= exp((ps - (psPrime / (1 - psPrime)) * (1 - ps)) / d);

		if ((sigma * sqrt(fabs(eigenvalues(n - 1)))) < lowerBound)
			sigma = lowerBound / sqrt(fabs(eigenvalues(n - 1)));
	};

	void updateCovariance(Individual& parent)
	{
		for (i = 0; i < n; i++) {
			if (ps < 0.44) pc(i) = (1 - cc) * pc(i)
				+ sqrt(cc * (2 - cc))
				* ((*this)[i] - (dynamic_cast<std::vector<double > & >(parent[0]))[i]) / sigma;
			else pc(i) = (1 - cc) * pc(i);
		}

		for (i = 0; i < n; i++) {
			for (j = 0; j < n; j++) {
				if (ps >= 0.44) C(i, j) = (1 - ccov) * C(i, j)
					+ ccov * (pc(i) * pc(j) + cc * (2 - cc) * C(i, j));
				else C(i, j) = (1 - ccov) * C(i, j)
					+ ccov * pc(i) * pc(j);
			}
		}
		myeig eig;
		eig.eigensymm(C, B, eigenvalues);

		needCovarianceUpdate = false;
	};

	void updateCovariance()
	{
		for (i = 0; i < n; i++) {
			if (ps < 0.44) pc(i) = (1 - cc) * pc(i) + sqrt(cc * (2 - cc)) * lastStep(i);
			else pc(i) = (1 - cc) * pc(i);
		}

		for (i = 0; i < n; i++) {
			for (j = 0; j < n; j++) {
				if (ps >= 0.44) C(i, j) = (1 - ccov) * C(i, j)
					+ ccov * (pc(i) * pc(j) + cc * (2 - cc) * C(i, j));
				else C(i, j) = (1 - ccov) * C(i, j)
					+ ccov * pc(i) * pc(j);
			}
		}

		myeig eig;
		eig.eigensymm(C, B, eigenvalues);

		needCovarianceUpdate = false;
	};

	const Array<double> &getC() const
	{
		return C;
	}

	double getSigma() const
	{
		return sigma;
	};

	double getCondition() const
	{
		return eigenvalues(0) / eigenvalues(n - 1);
	}

	void setLower(double lower)
	{
		lowerBound = lower;
	}

	bool covarianceUpdateNeeded()
	{
		return needCovarianceUpdate;
	}

	void putValues(Array<double> vals)
	{
		for (int i = 0; i<vals.dim(0); i++)
			(*this)[i] = vals(i);
	}

	Array<double> getValues()
	{
		Array<double> vals(this->size());
		//		vals.resize(n);
		for (int i = 0; i<this->size(); i++)
			vals(i) = (*this)[i];
		return vals;
	}

	int iParent;
	int iChild;
protected:
	Chromosome* clone() const
	{
		return new ChromosomeCMA(*this);
	}
	Chromosome* empty() const
	{
		return new ChromosomeCMA;
	}

public:
	unsigned n;		// Dimension // problem dimension / chromosome length
	unsigned lambda;      //	     // size of offspring population
	unsigned i, j, k;	//	     // iterators
	double sigma;		// delta     // the current global stepsize
	double psPrime;	//	     // target success probability; strategy constant for global stepsize update
	double cc;		// c	     // strategy constant for cma culmulation update
	double ccov;		//	     //	strategy constant for cma update
	double d;
	double cp;
	double ps;		// avsuccess // global stepsize culmulation path
	Array<double> pc;	// s	     // evo cumulation path
	Array2D<double> C;	//	     // the current covariance matrix
	Array2D<double> B;	//	     // eigenvector matrix C for offspring sampling
	Array<double> z;      // standard normally distributed mutation vector
	Array<double> eigenvalues;	     // eigenvalue vector of C for offspring sampling
	Array<double> lastStep;      //

	unsigned      noOffspringSinceUpdate;
	double        lambdaSucc;
	double        lowerBound;
	bool needCovarianceUpdate;

	vector<double>	rewards;
	vector<int>	tries;
	double	reward;
	int		ntry;
	int		status;		//parentArm = 1; childArm = 2;	else = 0;

	int	isChild;	//
	Array<double> weights;

	vector< MyRew >	MyRews;

	void ReStart()
	{
		for (int i = 0; i<n; i++)
		{
			for (int j = 0; j<n; j++)
			{
				if (i == j)
				{
					C(i, j) = 1;

				}
				else C(i, j) = 0;
			}
		}
		myeig eig;
		eig.eigensymm(C, B, eigenvalues);
	}

	double getAvrReward()
	{
		if (1)
		{
			double avr = 0;
			for (int i = 0; i<rewards.size(); i++)
				avr += rewards.at(i);
			if (rewards.size() == 0)	avr = 1e+10;
			else						avr /= rewards.size();
			return avr;
		}
		if (0)
		{
			int nrew = rewards.size();
			if (nrew == 0)	return 1e+10;
			weights.resize(nrew);
			double sum = 0;
			for (int i = 0; i<nrew; i++)
			{
				weights(i) = log(double(nrew + 0.5)) - log(double(1 + i));
				sum += weights(i);
			}
			for (int i = 0; i<nrew; i++)
				weights(i) /= sum;

			double avr = 0;
			for (int i = 0; i<nrew; i++)
				avr += weights(i)*(rewards.at(nrew - 1 - i));
			return avr;
		}

	}

	void UpdateMyRews(int UpToIter)
	{
		vector< MyRew >	NewRews;
		for (int i = 0; i<MyRews.size(); i++)
			if (MyRews.at(i).iter > UpToIter)
				NewRews.push_back(MyRews.at(i));
		MyRews = NewRews;
	}
	double GetMyAverageRew()
	{
		double avr = 0;
		for (int i = 0; i<MyRews.size(); i++)
			avr += MyRews.at(i).rew;
		if (MyRews.size() <= 0)	return avr = 1;//1e+10;
		else					avr = avr / double(MyRews.size());
		return avr;
	}

	double GetMyMaxRew()
	{
		double max = -1e+10;
		for (int i = 0; i<MyRews.size(); i++)
			if (MyRews.at(i).rew > max)	max = MyRews.at(i).rew;
		if (MyRews.size() <= 0)	return max = 1;//1e+10;
		return max;
	}

	int GetNTry()
	{
		int nTry = 0;
		for (int i = 0; i<MyRews.size(); i++)
			nTry += MyRews.at(i).isParentRew;
		return nTry;
	}
};

class PopulationMOO : public Population
{
public:
	PopulationMOO(unsigned n, const IndividualMOO& indmoo)
		: Population()
	{
		for (unsigned i = n; i--;)
		{
			IndividualMOO *dummy = new IndividualMOO(indmoo);
			vector< Individual * >::push_back(dummy);
		}
	}

	PopulationMOO(const Population& pop)
		: Population(pop)
	{
		for (unsigned i = pop.size(); i--;)
		{
			delete *(begin() + i);
			*(begin() + i) = new IndividualMOO(pop[i]);
		}
		ascending = pop.ascendingFitness();
	}

	~PopulationMOO()
	{ }

	unsigned size() const
	{
		return Population::size();
	}

	std::vector< Individual* >::iterator begin()
	{
		return vector< Individual * >::begin();
	}

	std::vector< Individual* >::iterator end()
	{
		return vector< Individual * >::end();
	}

	void resize(unsigned n)
	{
		unsigned i;
		unsigned s = size();

		if (n < s)
		{
			for (i = n; i < s; i++)
			{
				delete *(begin() + i);
			}
			vector< Individual * >::erase(begin() + n, end());
		}
		else if (n > s)
		{
			if (s > 0)
			{
				for (i = s; i < n; i++)
				{
					IndividualMOO *dummy = new IndividualMOO((*this)[0]);
					vector< Individual * >::push_back(dummy);
				}
			}
			else
			{
				for (i = s; i < n; i++)
				{
					IndividualMOO *dummy = new IndividualMOO;
					vector< Individual * >::push_back(dummy);
				}
			}
		}
	}

	IndividualMOO& operator [ ](unsigned i)
	{
		RANGE_CHECK(i < size())
			return *(static_cast< IndividualMOO * >(
			vector< Individual * >::operator[ ](i)));
	}
	const IndividualMOO& operator [ ](unsigned i) const
	{
		RANGE_CHECK(i < size())
			return *(static_cast< IndividualMOO * >(
			vector< Individual * >::operator[ ](i)));
	}
	PopulationMOO& operator = (const IndividualMOO & indmoo)
	{
		for (unsigned i = size(); i--;)
		{
			(*this)[i] = indmoo;
		}
		return *this;
	}
	PopulationMOO& operator = (const Individual & ind)
	{
		for (unsigned i = size(); i--;)
		{
			(*this)[i] = IndividualMOO(ind);
		}
		return *this;
	}
	PopulationMOO& operator = (const PopulationMOO& popmoo)
	{
		if (size() != popmoo.size())
		{
			cerr << "Error in TO-PM-044" << endl;
			throw SHARKEXCEPTION("size mismatch");
		}
		for (unsigned i = size(); i--;)
		{
			(*this)[i] = popmoo[i];
		}
		index = popmoo.index;
		ascending = popmoo.ascending;

		return *this;
	}
	PopulationMOO& operator = (const Population& pop)
	{
		if (size() != pop.size())
		{
			cerr << "Error in TO-PM-044" << endl;
			throw SHARKEXCEPTION("size mismatch");
		}
		for (unsigned i = size(); i--;)
		{
			(*this)[i] = pop[i];
		}
		ascending = pop.ascendingFitness();

		return *this;
	}
	bool operator == (const PopulationMOO& popmoo) const
	{
		if (size() == popmoo.size())
		{
			for (unsigned i = size(); i--;)
			{
				if ((*this)[i] != popmoo[i])
				{
					return false;
				}
			}

			return	index == popmoo.index        &&
				ascending == popmoo.ascending;
		}

		return false;
	}

	void setAscending(bool strategy)
	{
		if (strategy)
		{
			Population::setMinimize();
		}
		else
		{
			Population::setMaximize();
		}
	}

	void setNoOfObj(unsigned NOO)
	{
		for (unsigned i = size(); i--;)
		{
			(*this)[i].setNoOfObj(NOO);
		}
	}
	void setMOOFitness(double obj)
	{
		for (unsigned i = size(); i--;)
		{
			for (unsigned j = (*this)[i].getNoOfObj(); j--;)
			{
				(*this)[i].setMOOFitness(j, obj);
			}
		}
	}
	void setMOORank(unsigned MOOR)
	{
		for (unsigned i = size(); i--;)
		{
			(*this)[i].setMOORank(MOOR);
		}
	}
	void setMOOShare(double MOOS)
	{
		for (unsigned i = size(); i--;)
		{
			(*this)[i].setMOOShare(MOOS);
		}
	}

	int Dominate(unsigned i1, unsigned i2, bool unpenalized = false)
	{
		if ((*this)[i1].getNoOfObj() != (*this)[i2].getNoOfObj())
		{
			cerr << "\n ***** The number of objective functions are different in TO-PM-211 *****\n" << endl;
			throw SHARKEXCEPTION("wrong number of objectives");
		}
		unsigned NoOfObj = (*this)[i1].getNoOfObj();

		unsigned flag1 = 0;
		unsigned flag2 = 0;
		unsigned flag3 = 0;

		if (unpenalized)
		{
			for (unsigned i = NoOfObj; i--;)
			{
				if ((*this)[i1].getUnpenalizedMOOFitness(i) > (*this)[i2].getUnpenalizedMOOFitness(i))
				{
					flag1++;
				}
				else if ((*this)[i1].getUnpenalizedMOOFitness(i) < (*this)[i2].getUnpenalizedMOOFitness(i))
				{
					flag3++;
				}
				else
				{
					flag2++;
				}
			}
		}
		else
		{
			for (unsigned i = NoOfObj; i--;)
			{
				if ((*this)[i1].getMOOFitness(i) > (*this)[i2].getMOOFitness(i))
				{
					flag1++;
				}
				else if ((*this)[i1].getMOOFitness(i) < (*this)[i2].getMOOFitness(i))
				{
					flag3++;
				}
				else
				{
					flag2++;
				}
			}
		}
		if (!ascending)
		{
			std::swap(flag1, flag3);
		}

		// relationship
		if (flag1 + flag2 + flag3 != NoOfObj)
		{
			return 0; // abnormal
		}
		else if (flag3 == NoOfObj)
		{
			return 3; // i1 dominates i2 completely
		}
		else if (flag3 != 0 && flag1 == 0)
		{
			return 2; // i1 dominates i2 imcompletely
		}
		else if (flag2 == NoOfObj)
		{
			return 1; // i1 equals i2
		}
		else if (flag1 == NoOfObj)
		{
			return -3; // i2 dominates i1 completely
		}
		else if (flag1 != 0 && flag3 == 0)
		{
			return -2; // i2 dominates i1 imcompletely
		}
		else
		{
			return -1; // trade off
		}
	}

	void SMeasure(bool UnpenalizedFitness = false)
	{
		if (size() == 0) return;

		if ((*this)[0].getNoOfObj() != 2) throw SHARKEXCEPTION("The S-measure is implemented only for two objectives");

		SMeasureTwoObjectives(UnpenalizedFitness ? &IndividualMOO::getUnpenalizedMOOFitness : &IndividualMOO::getMOOFitness);
	}

	// this functional could be a lambda in C++11
	class LexicographicOrder
	{
	public:
		LexicographicOrder(PopulationMOO& S, double(IndividualMOO::*f)(unsigned))
			: m_S(S)
			, m_f(f)
		{ }

		bool operator () (std::size_t lhs, std::size_t rhs)
		{
			return (
				((m_S[lhs].*m_f)(0) < (m_S[rhs].*m_f)(0))
				|| (((m_S[lhs].*m_f)(0) == (m_S[rhs].*m_f)(0))
				&& ((m_S[lhs].*m_f)(1) < (m_S[rhs].*m_f)(1)))
				);
		}

	private:
		PopulationMOO& m_S;
		double(IndividualMOO::*m_f)(unsigned);
	};

	void SMeasureTwoObjectives(double(IndividualMOO::*f)(unsigned))
	{
		// lexicographic order of objective vectors
		PopulationMOO& S = *this;
		std::vector<std::size_t> order(size());
		for (std::size_t i = 0; i<size(); i++) order[i] = i;

		std::sort(order.begin(), order.end(), LexicographicOrder(S, f));

		// sweeping algorithm
		std::vector<std::size_t> T;    // invariant: S[T[r-1]].getMOORank() == r
		std::vector<double> prev_y;    // penultimate value of second objective in the given rank
		std::size_t s0 = 1;
		for (std::size_t i = 0; i<S.size(); i++)
		{
			std::size_t o = order[i];
			double x = (S[o].*f)(0);
			double y = (S[o].*f)(1);

			// handle duplicate objective vectors
			if (i > 0 && (S[order[i - 1]].*f)(0) == x && (S[order[i - 1]].*f)(1) == y)
			{
				S[o].setMOORank(S[order[i - 1]].getMOORank());
				continue;
			}

			// determine the front (MOO rank) in log_2(T.size()) steps
			std::size_t r = 0;
			for (std::size_t s = s0; s>0; s /= 2)
			{
				if (r + s < T.size() && y >= (S[T[r + s]].*f)(1)) r += s;
			}
			if (r < T.size() && y >= (S[T[r]].*f)(1)) r++;
			S[o].setMOORank(r + 1);

			// compute the contributing hypervolume within the rank,
			// extreme points get the maximal possible value
			if (r == T.size())
			{
				S[o].setMOOShare(MAXDOUBLE);
				T.push_back(o);
				prev_y.push_back(MAXDOUBLE);
				if (2 * s0 < T.size()) s0 *= 2;
			}
			else
			{
				double xx = (S[T[r]].*f)(0);
				double yy = (S[T[r]].*f)(1);
				S[o].setMOOShare(MAXDOUBLE);
				S[T[r]].setMOOShare((prev_y[r] == MAXDOUBLE) ? MAXDOUBLE : (x - xx) * (prev_y[r] - yy));
				T[r] = o;
				prev_y[r] = yy;
			}
		}
	}
	unsigned maxMOORank()
	{
		unsigned rank = 0;
		for (unsigned i = 0; i < size(); i++)
		{
			if ((*this)[i].getMOORank() > rank)
			{
				rank = (*this)[i].getMOORank();
			}
		}
		return rank;
	}

	// static comparison functions for std::sort
	static bool compareRankShare(Individual*const& pInd1, Individual*const& pInd2)
	{
		IndividualMOO* pI1 = (IndividualMOO*)pInd1;
		IndividualMOO* pI2 = (IndividualMOO*)pInd2;

		unsigned Rank1 = pI1->getMOORank();
		unsigned Rank2 = pI2->getMOORank();
		double Share1 = pI1->getMOOShare();
		double Share2 = pI2->getMOOShare();

		return ((Rank1 < Rank2) || (Rank1 == Rank2 && Share1 > Share2));
	}
	static bool compareScaledFitnessRankShare(Individual*const& pInd1, Individual*const& pInd2);
};


#endif
