
#ifndef _mathutil_H_
#define _mathutil_H_


#include <cstdio>
#include <cstdlib>
#include <cassert>
#include <cmath>
#include <vector>
#include <algorithm>


////////////////////////////////////////////////////////////
// Mathematical (real) vector supporting the most common
// linear algebra operations.
//
class Vector
{
public:
	typedef std::vector<double>::iterator iterator;
	typedef std::vector<double>::const_iterator const_iterator;

	Vector()
	: m_data(0)
	{ }

	explicit Vector(std::size_t dim)
	: m_data(dim)
	{
		for (std::size_t i=0; i<dim; i++) m_data[i] = 0.0;
	}

	explicit Vector(int dim)
	: m_data(dim)
	{
		assert(dim >= 0);
		for (std::size_t i=0; i<(size_t)dim; i++) m_data[i] = 0.0;
	}

	explicit Vector(unsigned int dim)
	: m_data(dim)
	{
		for (std::size_t i=0; i<(size_t)dim; i++) m_data[i] = 0.0;
	}

	Vector(std::size_t dim, double value)
	: m_data(dim)
	{
		for (std::size_t i=0; i<dim; i++) m_data[i] = value;
	}

	Vector(std::size_t dim, const double* data)
	: m_data(dim)
	{
		for (std::size_t i=0; i<dim; i++) m_data[i] = data[i];
	}

	Vector(Vector const& other)
	: m_data(other.m_data)
	{ }

	template <class VECTOR>
	Vector(VECTOR const& other)
	: m_data(other.size())
	{
		for (std::size_t i=0; i<other.size(); i++) m_data[i] = other[i];
	}

	void operator = (double value)
	{ for (std::size_t i=0; i<m_data.size(); i++) m_data[i] = value; }

	void operator = (Vector const& other)
	{ m_data = other.m_data; }

	template <class VECTOR>
	void operator = (VECTOR const& other)
	{
		std::size_t i, ic = other.size();
		m_data.resize(ic);
		for (i=0; i<ic; i++) m_data[i] = other[i];
	}

	inline iterator begin()
	{ return m_data.begin(); }
	inline const_iterator begin() const
	{ return m_data.begin(); }
	inline iterator end()
	{ return m_data.end(); }
	inline const_iterator end() const
	{ return m_data.end(); }

	inline std::size_t dim() const
	{ return m_data.size(); }
	inline std::size_t size() const
	{ return m_data.size(); }
	inline double& operator [] (std::size_t index)
	{ return m_data[index]; }
	inline double const& operator [] (std::size_t index) const
	{ return m_data[index]; }
	inline double& operator () (std::size_t index)
	{ return m_data[index]; }
	inline double const& operator () (std::size_t index) const
	{ return m_data[index]; }

	inline void operator += (Vector const& other)
	{
		assert(other.dim() == dim());
		std::size_t i, ic = size();
		for (i=0; i<ic; i++) m_data[i] += other.m_data[i];
	}
	inline void operator -= (Vector const& other)
	{
		assert(other.dim() == dim());
		std::size_t i, ic = size();
		for (i=0; i<ic; i++) m_data[i] -= other.m_data[i];
	}
	inline void operator *= (double scalar)
	{
		std::size_t i, ic = size();
		for (i=0; i<ic; i++) m_data[i] *= scalar;
	}
	inline void operator /= (double scalar)
	{
		std::size_t i, ic = size();
		for (i=0; i<ic; i++) m_data[i] /= scalar;
	}
	inline Vector operator + (Vector const& other) const
	{
		assert(other.dim() == dim());
		Vector ret(*this);
		ret += other;
		return ret;
	}
	inline Vector operator - () const
	{
		std::size_t i, ic = size();
		Vector ret(ic);
		for (i=0; i<ic; i++) ret[i] = -m_data[i];
		return ret;
	}
	inline Vector operator - (Vector const& other) const
	{
		assert(other.dim() == dim());
		Vector ret(*this);
		ret -= other;
		return ret;
	}
	inline Vector operator * (double scalar) const
	{
		Vector ret(*this);
		ret *= scalar;
		return ret;
	}
	inline Vector operator / (double scalar) const
	{
		Vector ret(*this);
		ret /= scalar;
		return ret;
	}
	inline double operator * (Vector const& other) const
	{
		assert(other.dim() == dim());
		double ret = 0.0;
		std::size_t i, ic = size();
		for (i=0; i<ic; i++) ret += m_data[i] * other.m_data[i];
		return ret;
	}

	// resize, keep old content, fill with zeros
	inline void resize(std::size_t dim)
	{
		std::size_t i, olddim = m_data.size();
		m_data.resize(dim);
		for (i=olddim; i<dim; i++) m_data[i] = 0.0;
	}

	inline Vector sub(std::size_t start, std::size_t num) const
	{
		Vector ret(num);
		for (std::size_t i=0; i<num; i++) ret(i) = m_data[start + i];
		return ret;
	}

	inline double onenorm() const
	{
		double ret = 0.0;
		std::size_t i, ic = m_data.size();
		for (i=0; i<ic; i++) ret += std::fabs(m_data[i]);
		return ret;
	}
	inline double twonorm2() const
	{ return (*this) * (*this); }
	inline double twonorm() const
	{ return std::sqrt(twonorm2()); }
	inline void normalize()
	{ (*this) /= twonorm(); }

	inline double max() const
	{
		std::size_t i, ic = m_data.size();
		if (ic == 0) return -1e100;
		double ret = m_data[0];
		for (i=1; i<ic; i++)
		{
			double d = m_data[i];
			if (d > ret) ret = d;
		}
		return ret;
	}
	inline double min() const
	{
		std::size_t i, ic = m_data.size();
		if (ic == 0) return +1e100;
		double ret = m_data[0];
		for (i=1; i<ic; i++)
		{
			double d = m_data[i];
			if (d < ret) ret = d;
		}
		return ret;
	}

	inline void print(const char* name = nullptr) const
	{
		if (name) printf("%s:\n", name);
		for (std::size_t i=0; i<dim(); i++) printf(" %g", (*this)(i));
		printf("\n");
	}

protected:
	std::vector<double> m_data;
};


////////////////////////////////////////////////////////////
// Real matrix, supporting basic linear algebra operations
// as well as eigen-decomposition and matrix exponential.
//
// NOTE: Eigen-decomposition and exponential exponential
//       assume a SYMMETRIC square matrix. For non-symmetric
//       matrices the result is undefined!
//
class Matrix
{
private:
	class CmpIndex
	{
	public:
		CmpIndex(Vector const& lambda, std::vector<std::size_t>& index)
		: m_lambda(lambda)
		, m_index(index)
		{ }

		inline bool operator ()(std::size_t i1, std::size_t i2) const
		{ return (m_lambda[i1] > m_lambda[i2]); }

	protected:
		Vector const& m_lambda;
		std::vector<std::size_t>& m_index;
	};

public:
	typedef std::vector<double>::iterator iterator;
	typedef std::vector<double>::const_iterator const_iterator;

	Matrix()
	: m_rows(0)
	, m_cols(0)
	, m_data(0)
	{ }

	Matrix(std::size_t rows, std::size_t cols)
	: m_rows(rows)
	, m_cols(cols)
	, m_data(rows * cols)
	{
		std::size_t i, ic = m_data.size();
		for (i=0; i<ic; i++) m_data[i] = 0.0;
	}

	void operator = (Matrix const& other)
	{
		m_rows = other.m_rows;
		m_cols = other.m_cols;
		m_data = other.m_data;
	}

	inline std::size_t rows() const
	{ return m_rows; }
	inline std::size_t cols() const
	{ return m_cols; }
	inline std::size_t size1() const
	{ return rows(); }
	inline std::size_t size2() const
	{ return cols(); }
	inline double& operator () (std::size_t y, std::size_t x)
	{
		assert(y < m_rows && x < m_cols);
		return m_data[x + m_cols * y];
	}
	inline double const& operator () (std::size_t y, std::size_t x) const
	{
		assert(y < m_rows && x < m_cols);
		return m_data[x + m_cols * y];
	}

	inline iterator begin()
	{ return m_data.begin(); }
	inline const_iterator begin() const
	{ return m_data.begin(); }
	inline iterator end()
	{ return m_data.end(); }
	inline const_iterator end() const
	{ return m_data.end(); }

	inline void operator += (Matrix const& other)
	{
		assert(m_rows == other.rows() && m_cols == other.cols());
		std::size_t i, ic = m_rows * m_cols;
		for (i=0; i<ic; i++) m_data[i] += other.m_data[i];
	}
	inline void operator -= (Matrix const& other)
	{
		assert(m_rows == other.rows() && m_cols == other.cols());
		std::size_t i, ic = m_rows * m_cols;
		for (i=0; i<ic; i++) m_data[i] -= other.m_data[i];
	}
	inline void operator *= (double scalar)
	{
		std::size_t i, ic = m_rows * m_cols;
		for (i=0; i<ic; i++) m_data[i] *= scalar;
	}
	inline void operator /= (double scalar)
	{
		std::size_t i, ic = m_rows * m_cols;
		for (i=0; i<ic; i++) m_data[i] /= scalar;
	}
	inline Matrix operator + (Matrix const& other) const
	{
		assert(m_rows == other.rows() && m_cols == other.cols());
		Matrix ret(*this);
		ret += other;
		return ret;
	}
	inline Matrix operator - (Matrix const& other) const
	{
		assert(m_rows == other.rows() && m_cols == other.cols());
		Matrix ret(*this);
		ret -= other;
		return ret;
	}
	inline Matrix operator * (double scalar) const
	{
		Matrix ret(*this);
		ret *= scalar;
		return ret;
	}
	inline Matrix operator / (double scalar) const
	{
		Matrix ret(*this);
		ret /= scalar;
		return ret;
	}
	inline Matrix operator * (Matrix const& other) const
	{
		assert(m_cols == other.rows());
		Matrix ret(m_rows, other.cols());
		std::size_t i, j, k;
		for (i=0; i<m_rows; i++)
		{
			for (j=0; j<other.cols(); j++)
			{
				double value = 0.0;
				for (k=0; k<m_cols; k++)
				{
					value += (*this)(i, k) * other(k, j);
				}
				ret(i, j) = value;
			}
		}
		return ret;
	}
	inline Vector operator * (Vector const& other) const
	{
		assert(m_cols == other.dim());
		Vector ret(m_rows);
		std::size_t i, j;
		for (i=0; i<m_rows; i++)
		{
			double value = 0.0;
			for (j=0; j<m_cols; j++)
			{
				value += (*this)(i, j) * other[j];
			}
			ret[i] = value;
		}
		return ret;
	}

	inline void operator *= (Matrix const& other)
	{ *this = Matrix::operator *((Matrix const&)other); }

	// resize and set entries to zeros
	inline void resize(std::size_t rows, std::size_t cols)
	{
		m_rows = rows;
		m_cols = cols;
		std::size_t i, ic = rows * cols;
		m_data.resize(ic);
		for (i=0; i<ic; i++) m_data[i] = 0.0;
	}

	inline Matrix transpose() const
	{
		Matrix ret(m_cols, m_rows);
		std::size_t i, j;
		for (i=0; i<m_cols; i++)
			for (j=0; j<m_rows; j++)
				ret(i, j) = (*this)(j, i);
		return ret;
	}

	inline void print(const char* name = NULL) const
	{
		if (name != NULL) printf("%s:\n", name);
		std::size_t i, j;
		for (i=0; i<m_rows; i++)
		{
			for (j=0; j<m_cols; j++) printf(" %g", (*this)(i, j));
			printf("\n");
		}
	}

	// eigen decomposition
	void eig(Matrix& U, Vector& lambda, std::size_t iter = 200, bool ignoreError = true) const;

	// compose matrix from eigen vectors and eigen values
	static Matrix fromEig(Matrix const& U, Vector const& lambda)
	{
		assert(U.rows() == U.cols() && U.rows() == lambda.dim());

		std::size_t n = lambda.size();
		Matrix ret(n, n);
		std::size_t i, j, k;
		for (i = 0; i < n; i++)
		{
			for (j = 0; j < n; j++)
			{
				double sum = 0.0;
				for (k = 0; k < n; k++) sum += U(i, k) * lambda[k] * U(j, k);
				ret(i, j) = sum;
			}
		}
		return ret;
	}

	// matrix exponential
	inline Matrix exp() const
	{
		assert(m_rows == m_cols);
		Matrix U;
		Vector lambda;
		eig(U, lambda);
		for (std::size_t i=0; i<m_rows; i++) lambda[i] = std::exp(lambda[i]);
		return fromEig(U, lambda);
	}

	// return I
	static Matrix identity(std::size_t dim)
	{
		Matrix ret(dim, dim);
		for (std::size_t i=0; i<dim; i++) ret(i, i) = 1.0;
		return ret;
	}

	// return v * w^T
	static Matrix outerProduct(Vector const& v, Vector const& w)
	{
		std::size_t i, j;
		Matrix ret(v.dim(), w.dim());
		for (i=0; i<v.dim(); i++)
		{
			for (j=0; j<w.dim(); j++)
			{
				ret(i, j) = v[i] * w[j];
			}
		}
		return ret;
	}

	// return v * v^T
	static Matrix outerProduct(Vector const& v)
	{
		std::size_t i, j, dim = v.size();
		Matrix ret(dim, dim);
		for (i=0; i<dim; i++)
		{
			for (j=0; j<i; j++)
			{
				double p = v[i] * v[j];
				ret(i, j) = p;
				ret(j, i) = p;
			}
			ret(i, i) = v[i] * v[i];
		}
		return ret;
	}

	static Matrix diag(Vector const& vec)
	{
		std::size_t i, ic = vec.dim();
		Matrix ret(ic, ic);
		for (i=0; i<ic; i++) ret(i, i) = vec(i);
		return ret;
	}

protected:
	std::size_t m_rows;
	std::size_t m_cols;
	std::vector<double> m_data;
};


Vector operator * (double lhs, Vector rhs);
Matrix operator * (double lhs, Matrix rhs);


#endif
