
#include "asyncES.h"
#include <vector>
#include <thread>
#include <chrono>
#include <iostream>


using namespace std;


// objective function type
typedef double (*Objective)(Vector const&);

// objective function instance
double rosenbrock(Vector const& x)
{
	// pretend that the computation takes a while
	this_thread::sleep_for(chrono::milliseconds(rand() % 10));

	// compute the Rosenbrock function
	size_t dim = x.size();
	double sum = 0.0;
	for (size_t i=1; i<dim; i++)
	{
		double y = x(i - 1);
		double z = x(i);
		double a = (1.0 - y);
		double b = z - y * y;
		sum += a * a + 100.0 * b * b;
	}
	return sum;
}

// global variable (sic!) controlling the worker threads
bool running = true;

// worker thread
void worker(AsyncES* es, Objective f)
{
	while (running)
	{
		Individual ind = es->generatePoint();
		double fitness = f(ind.point());
		ind.setFitness(fitness);
		es->evaluationReady(ind);
	}
}

// solve the 10-dimensional Rosenbrock problem
int main()
{
	// problem definition
	Objective obj = rosenbrock;   // problem to solve
	unsigned int dim = 10;        // problem dimension
	double target = 1e-10;        // target fitness
	unsigned int cpus = 8;        // number of CPUs

	cout << "problem dimension: " << dim << endl;
	cout << "target objective value: " << target << endl;
	cout << "number of threads: " << cpus << endl;

	// define initial search distribution and create asynchronous algorithm object
	Vector x0(dim);
	double sigma = 1.0 / dim;
	AsyncES es(x0, sigma, cpus);

	// create worker threads
	vector<thread*> pool;
	for (size_t i=0; i<cpus; i++)
	{
		pool.push_back(new thread(worker, &es, obj));
	}

	// wait for good enough fitness value
	cout << "optimizing ... please wait ..." << endl;
	while (es.bestFitness() > target)
	{
		this_thread::sleep_for(chrono::milliseconds(10));
	}

	// wait for threads to finish
	running = false;
	for (size_t i=0; i<cpus; i++)
	{
		pool[i]->join();
		delete pool[i];
	}

	// report best fitness
	cout << "fitness evaluations: " << es.evaluations() << endl;
	cout << "best fitness value: " << es.bestFitness() << endl;
}
