
#ifndef _asyncES_H_
#define _asyncES_H_


#include "mathutil.h"
#include <vector>
#include <mutex>


class AsyncES;


// Representation of a search point, possibly with its fitness value
// (if already evaluated). Individuals are created by calling
// AsyncES::generatePoint().
class Individual
{
private:
	friend class AsyncES;

	Individual(Vector const& g, Vector const& x)
	: m_g(g)
	, m_x(x)
	, m_fitness(1e100)
	{ }

public:
	Individual(Individual const& other)
	: m_g(other.m_g)
	, m_x(other.m_x)
	, m_fitness(other.m_fitness)
	{ }

	~Individual() { }

	Vector const& point() const
	{ return m_x; }

	bool isEvaluated() const
	{ return (m_fitness < 1e100); }
	double fitness() const
	{ return m_fitness; }
	void setFitness(double f)
	{ m_fitness = f; }

private:
	Vector m_g;
	Vector m_x;
	double m_fitness;
};


// Asynchronous xNES algorithm.
// The algorithm supports two operations, namely the generation of a new
// individual (to be evaluated asynchronously), and the update of the
// algorithm state based on a readily evaluated individual. As soon as a
// processor becomes idle it should call the generatePoint() method to
// obtain a new point for evaluation. As soon as it is done it should
// call evaluationReady(), followed by generatePoint().
class AsyncES
{
public:
	// Initialize the asynchronous search algorithm by providing mean
	// and standard deviation of the initial search distribution as well
	// as the number of CPUs (threads) used for asynchronous evaluation.
	// This number can be a rough estimate. The algorithm does not keep
	// track of this number, it may even change during the course of a
	// run. It is only used to estimate a suitable learning rate.
	AsyncES(Vector const& mean, double stddev, unsigned int numCPUs);
	~AsyncES();

	Individual generatePoint();
	void evaluationReady(Individual const& ind);

	// Mean vector of the distribution. At each time this point is
	// recommended as a "return" value, it is often better than the best
	// individual seen so far.
	Vector mean() const
	{ return m_mean; }

	Vector bestPoint() const
	{ return m_bestPoint; }
	double bestFitness() const
	{ return m_bestFitness; }

	unsigned int evaluations() const
	{ return m_evaluations; }

private:
	static std::vector<double> utilities(size_t size);

	mutable std::mutex m_mutex;

	Vector m_mean;                   // mean of the Gaussian
	Matrix m_A;                      // transformation matrix (factor of the covariance matrix)
	std::vector<Individual> m_pop;   // population (recent history)

	size_t m_lambda;                 // "virtual" population size
	double m_eta_mean;               // mean learning rate
	double m_eta_A;                  // covariance learning rate

	unsigned int m_evaluations;      // number of fitness evaluations
	Vector m_bestPoint;              // best point seen so far
	double m_bestFitness;            // fitness of the best point
};


#endif
