
#include "asyncES.h"
#include <stdexcept>


using namespace std;


#define SUM_ZERO_UTILITY


double uniform()
{
	// this function would profit from a better implementation...
	return (rand() + 0.5) / (RAND_MAX + 1.0);
}

double gauss()
{
	return sin(2.0 * M_PI * uniform()) * std::sqrt(-2.0 * std::log(uniform()));
}


////////////////////////////////////////////////////////////


// comparator on pointers for sorting by fitness
bool comp_fitness(const Individual* pi1, const Individual* pi2)
{ return (pi1->fitness() < pi2->fitness()); }


AsyncES::AsyncES(Vector const& mean, double stddev, unsigned int numCPUs)
: m_mean(mean)
, m_A(stddev * Matrix::identity(mean.size()))
, m_lambda((size_t)(4 + floor(3.0 * log((double)m_mean.size()))))
, m_eta_mean(pow(2.0 / 3.0, (2.0 * numCPUs) / (m_lambda * m_mean.size())))
, m_eta_A(m_eta_mean * 0.6 * (3.0 + log((double)mean.size())) / ((double)mean.size() * sqrt((double)mean.size())))
, m_evaluations(0)
, m_bestFitness(1e100)
{ }

AsyncES::~AsyncES()
{ }


Individual AsyncES::generatePoint()
{
	std::lock_guard<std::mutex> guard(m_mutex);

	Vector g(m_mean.size());
	for (unsigned int i=0; i<m_mean.size(); i++) g(i) = gauss();
	Vector x = m_mean + m_A * g;

	return Individual(g, x);
}

void AsyncES::evaluationReady(Individual const& ind)
{
	if (! ind.isEvaluated()) throw runtime_error("[AsyncES::evaluationReady] individual is not yet evaluated; call individual.setFitness(...) first");

	std::lock_guard<std::mutex> guard(m_mutex);

	// replace oldest individual with the newcomer
	m_evaluations++;
	m_pop.push_back(ind);
	if (m_pop.size() > m_lambda) m_pop.erase(m_pop.begin());

	// keep track of best point seen so far
	if (ind.fitness() < m_bestFitness)
	{
		m_bestPoint = ind.point();
		m_bestFitness = ind.fitness();
	}

	// check whether there is sufficient context
	size_t size = m_pop.size();
	if (size <= 2) return;

	// sort population by fitness
	vector<Individual*> sub(size);
	for (size_t i=0; i<size; i++) sub[i] = &m_pop[i];
	sort(sub.begin(), sub.end(), comp_fitness);

	// compute the stochastic natural gradient
	size_t dim = m_mean.size();
	Vector G_delta(dim);
	Matrix G_M(dim, dim);
	vector<double> util = utilities(size);
	for (size_t i=0; i<size; i++)
	{
		Vector const& g = sub[i]->m_g;
		G_delta += util[i] * g;
		Matrix tmp = Matrix::outerProduct(g);
		for (size_t j=0; j<dim; j++) tmp(j, j) -= 1.0;
		G_M += util[i] * tmp;
	}

	// update the algorithm state
	const double factor = (double)size / (double)(m_lambda * m_lambda);
	m_mean += m_A * ((factor * m_eta_mean) * G_delta);
	m_A = m_A * ((0.5 * factor * m_eta_A) * G_M).exp();
}

vector<double> AsyncES::utilities(size_t size)
{
	vector<double> util(size);
	const double mid = 0.5 * size;
	double sum = 0.0;
	for (size_t i=0; i<size; i++)
	{
		util[i] = std::max(0.0, (log(mid) - log(i + 1.0)));
		sum += util[i];
	}
	for (size_t i=0; i<size; i++)
	{
		util[i] /= sum;
#ifdef SUM_ZERO_UTILITY
		util[i] -= 1.0 / size;
#endif
	}
	return util;
}
